using System;
using System.IO;
using System.Collections.Generic;
using Latino;
using Latino.Model;

namespace SvmPosTagger
{
    class Program
    {
        static bool m_verbose
            = false;

        static void OutputHelp()
        {
            Console.WriteLine("*** SVM POS Tagger 1.0 - Tagging Module ***");
            //Console.WriteLine("(SOURCEFORGE.NET URL)");
            Console.WriteLine();
            Console.WriteLine("Usage:");
            Console.WriteLine("SvmPosTaggerTag [<options>] <corpus_file_name> <model_file_name>");
            Console.WriteLine("  <tagged_corpus_file_name>");
            Console.WriteLine();
            Console.WriteLine("<options>:                 See below.");
            Console.WriteLine("<corpus_file_name>:        Corpus for tagging (input).");
            Console.WriteLine("<model_file_name>:         Trained model (input).");
            Console.WriteLine("<tagged_corpus_file_name>: Tagged corpus (output).");
            Console.WriteLine();
            Console.WriteLine("Options:");
            Console.WriteLine("-v Verbose.");
            Console.WriteLine("   (default: not set; quiet mode)");
        }

        static void Verbose(string text, params object[] args)
        {
            if (m_verbose)
            {
                Console.Write(text, args);
            }
        }

        static bool ParseParams(string[] args, ref bool verbose, ref string corpus_file_name, ref string model_file_name, ref string tagged_corpus_file_name)
        {
            // parse
            for (int i = 0; i < args.Length - 3; i++)
            {
                string arg_lwr = args[i].ToLower();
                if (arg_lwr == "-v")
                {
                    verbose = true;
                }
                else
                {
                    Console.WriteLine("*** Invalid option {0}.\r\n", args[i]);
                    OutputHelp();
                    return false;
                }
            }
            // check file names
            corpus_file_name = args[args.Length - 3];
            model_file_name = args[args.Length - 2];
            tagged_corpus_file_name = args[args.Length - 1];
            if (!Utils.VerifyFileName(corpus_file_name, /*must_exist=*/true))
            {
                Console.WriteLine("*** Invalid corpus file name or file not found.\r\n");
                OutputHelp();
                return false;
            }
            if (!Utils.VerifyFileName(model_file_name, /*must_exist=*/true))
            {
                Console.WriteLine("*** Invalid model file name or file not found.\r\n");
                OutputHelp();
                return false;
            }
            if (!Utils.VerifyFileName(tagged_corpus_file_name, /*must_exist=*/false))
            {
                Console.WriteLine("*** Invalid tagged corpus file name.\r\n");
                OutputHelp();
                return false;
            }
            return true;
        }

        static void Main(string[] args)
        {
            try
            {
                if (args.Length < 3)
                {
                    OutputHelp();
                }
                else
                {                    
                    string corpus_file_name = null, model_file_name = null, tagged_corpus_file_name = null;
                    if (ParseParams(args, ref m_verbose, ref corpus_file_name, ref model_file_name, ref tagged_corpus_file_name))
                    {
                        Verbose("Loading models ...\r\n");
                        string model_dir = new FileInfo(model_file_name).DirectoryName;
                        BinarySerializer reader = new BinarySerializer(model_file_name, FileMode.Open);
                        reader.DataDir = model_dir;
                        WordDictionary dictionary = new WordDictionary(reader);
                        Dictionary<string, int> feature_space = Utils.LoadDictionary<string, int>(reader);
                        ModelIndex kw_model = new ModelIndex(reader);
                        PosModel uw_model = new PosModel(reader);
                        reader.Close();
                        Verbose("Loading test corpus ...\r\n");
                        Corpus corpus = new Corpus();
                        corpus.LoadFromFile(corpus_file_name, /*create_dictionary=*/false);
                        Verbose("Tagging test corpus ...\r\n");
                        int kw_correct_count = 0;
                        int kw_count = 0;
                        int uw_correct_count = 0;
                        int uw_count = 0;
                        for (int i = 0; i < corpus.TaggedWords.Count; i++)
                        {
                            Verbose("{0} / {1}\r", i + 1, corpus.TaggedWords.Count);
                            SparseVector2<double> feature_vector = corpus.GenerateFeatureVector(i, feature_space, /*extend_feature_space=*/false, dictionary);
                            string real_tag = corpus.TaggedWords[i].Tag;
                            if (dictionary.Contains(corpus.TaggedWords[i].WordL)) // tag known word
                            {
                                string tag = kw_model.Classify(feature_vector, dictionary.GetTags(corpus.TaggedWords[i].WordL));
                                if (tag == real_tag) { kw_correct_count++; }
                                corpus.TaggedWords[i].Tag = tag;
                                if (real_tag != null) { kw_count++; }
                            }
                            else // tag unknown word
                            {
                                Classification<string> classification = uw_model.Classify(feature_vector);
                                string tag = classification.BestClassLabel;
                                if (tag == real_tag) { uw_correct_count++; }
                                corpus.TaggedWords[i].Tag = tag;
                                if (real_tag != null) { uw_count++; }
                            }
                        }
                        Verbose("\r\n");
                        Verbose("Writing tagged corpus ...\r\n");
                        corpus.SaveIntoFile(tagged_corpus_file_name);
                        Verbose("Done.\r\n");
                        if (kw_count > 0)
                        {
                            Verbose("Accuracy on known words: {0} / {1}, {2}%\r\n",
                                kw_correct_count,
                                kw_count,
                                ((double)kw_correct_count / (double)kw_count * 100.0).ToString("0.00"));
                        }
                        if (uw_count > 0)
                        {
                            Verbose("Accuracy on unknown words: {0} / {1}, {2}%\r\n",
                                uw_correct_count,
                                uw_count,
                                ((double)uw_correct_count / (double)uw_count * 100.0).ToString("0.00"));
                        }
                        if (kw_count + uw_count > 0)
                        {
                            Verbose("Overall accuracy: {0} / {1}, {2}%\r\n",
                                kw_correct_count + uw_correct_count,
                                kw_count + uw_count,
                                ((double)(kw_correct_count + uw_correct_count) / (double)(kw_count + uw_count) * 100.0).ToString("0.00"));
                        }
                    }                    
                }
            }
            catch (Exception exception)
            {
                Console.WriteLine("*** Unexpected error occurred. Details: {0}\r\n{1}", exception, exception.StackTrace);
            }
        }
    }
}
