/*==========================================================================;
 *
 *  (c) 2007-08 JSI.  All rights reserved.
 *
 *  File:          Utils.cs
 *  Version:       1.0
 *  Desc:		   Latino utilities
 *  Author:        Miha Grcar
 *  Created on:    Nov-2007
 *  Last modified: May-2008
 *  Revision:      May-2008
 *
 ***************************************************************************/

using System;
using System.IO;
using System.Text;
using System.Diagnostics;
using System.Collections.Generic;
using System.Security.Cryptography;
using System.Collections;

namespace Latino
{
    /* .-----------------------------------------------------------------------
       |
       |  Static class Utils
       |
       '-----------------------------------------------------------------------
    */
    public static class Utils
    {
        [Conditional("VERBOSE")]
        internal static void Verbose(object sender, string format, params object[] args)
        {
            if (sender is Type)
            {
                Console.WriteLine(String.Format("[{0}] {1}", sender, format), args);
            }
            else
            {
                Console.WriteLine(String.Format("[{0} {1}] {2}", sender.GetType(), sender.GetHashCode(), format), args);
            }
        }
        [Conditional("THROW_EXCEPTIONS")]
        public static void ThrowException(Exception exception)
        {
            if (exception != null) { throw exception; }
        }
        internal static bool IsFiniteNumber(params double[] values)
        {
            foreach (double val in values)
            {
                if (double.IsInfinity(val) || double.IsNaN(val)) { return false; }
            }
            return true;
        }
        internal static bool IsFiniteNumber(params float[] values)
        {
            foreach (float val in values)
            {
                if (float.IsInfinity(val) || float.IsNaN(val)) { return false; }
            }
            return true;
        }
        public static bool VerifyFileName(string file_name, bool must_exist)
        {
            try
            {
                return must_exist ? new FileInfo(file_name).Exists : true;
            }
            catch 
            {
                return false; 
            }            
        }
        public static bool VerifyPathName(string path_name, bool must_exist)
        {
            try
            {
                return must_exist ? new DirectoryInfo(path_name).Exists : true;
            }
            catch
            {
                return false;
            }
        }
        public static string ReadTextFile(string file_name)
        {
            return ReadTextFile(file_name, Encoding.UTF8); // throws InvalidArgumentValueException, OutOfMemoryException, IOException
        }
        public static string ReadTextFile(string file_name, Encoding enc)
        {
            ThrowException(!VerifyFileName(file_name, /*must_exist=*/true) ? new InvalidArgumentValueException("file_name") : null);
            StreamReader reader = new StreamReader(file_name, enc);
            string text = reader.ReadToEnd(); // throws OutOfMemoryException, IOException
            reader.Close();
            return text;
        }
        // public static void WriteTextFile(string file_name, string text, Encoding enc);
        public static object Clone(object obj, bool deep_clone)
        {
            if (deep_clone && obj is IDeeplyCloneable)
            {
                return ((IDeeplyCloneable)obj).DeeplyClone();
            }
            else if (obj is ICloneable)
            {
                return ((ICloneable)obj).Clone();
            }
            else
            {
                return obj;
            }
        }
        public static bool ObjectEquals(object obj_1, object obj_2, bool deep_cmp)
        {
            if (obj_1 == null && obj_2 == null) { return true; }
            else if (obj_1 == null || obj_2 == null) { return false; }
            else if (!obj_1.GetType().Equals(obj_2.GetType())) { return false; }
            else if (deep_cmp && obj_1 is IContentEquatable)
            {
                return ((IContentEquatable)obj_1).ContentEquals(obj_2);
            }
            else
            {
                return obj_1.Equals(obj_2);
            }
        }
        public static int GetHashCode(object obj)
        {
            ThrowException(obj == null ? new ArgumentNullException("obj") : null);
            if (obj is ISerializable)
            {
                BinarySerializer mem_ser = new BinarySerializer();
                ((ISerializable)obj).Save(mem_ser);
                byte[] buffer = ((MemoryStream)mem_ser.Stream).GetBuffer();
                MD5CryptoServiceProvider hash_algo = new MD5CryptoServiceProvider();
                Guid md5_hash = new Guid(hash_algo.ComputeHash(buffer));
                return md5_hash.GetHashCode();
            }
            else
            {
                return obj.GetHashCode();
            }
        }
        public static void SaveDictionary<KeyT, ValT>(Dictionary<KeyT, ValT> dict, BinarySerializer writer)
        {
            Utils.ThrowException(dict == null ? new ArgumentNullException("dict") : null);
            Utils.ThrowException(writer == null ? new ArgumentNullException("writer") : null);
            ArrayList<Pair2<KeyT, ValT>> tmp = new ArrayList<Pair2<KeyT, ValT>>();
            foreach (KeyValuePair<KeyT, ValT> item in dict)
            {
                tmp.Add(new Pair2<KeyT, ValT>(item.Key, item.Value));
            }
            tmp.Save(writer); // throws serialization-related exceptions        
        }
        public static Dictionary<KeyT, ValT> LoadDictionary<KeyT, ValT>(BinarySerializer reader)
        {
            return LoadDictionary<KeyT, ValT>(reader, /*comparer=*/null); // throws ArgumentNullException, serialization-related exceptions
        }
        public static Dictionary<KeyT, ValT> LoadDictionary<KeyT, ValT>(BinarySerializer reader, IEqualityComparer<KeyT> comparer)
        {
            Utils.ThrowException(reader == null ? new ArgumentNullException("reader") : null);
            ArrayList<Pair2<KeyT, ValT>> tmp = new ArrayList<Pair2<KeyT, ValT>>(reader); // throws serialization-related exceptions
            Dictionary<KeyT, ValT> dict = new Dictionary<KeyT, ValT>(comparer);
            foreach (Pair2<KeyT, ValT> item in tmp)
            {
                dict.Add(item.First, item.Second);
            }
            return dict;
        }
    }

    /* .-----------------------------------------------------------------------
       |
       |  Class GenericEqualityComparer<T>
       |
       '-----------------------------------------------------------------------
    */
    public class GenericEqualityComparer<T> : IEqualityComparer<T>, IEqualityComparer where T : ISerializable
    {
        public bool Equals(T x, T y)
        {
            return Utils.ObjectEquals(x, y, /*deep_cmp=*/true);
        }
        public int GetHashCode(T obj)
        {
            return Utils.GetHashCode(obj); // throws ArgumentNullException
        }
        bool IEqualityComparer.Equals(object x, object y)
        {
            Utils.ThrowException((x != null && !(x is T)) ? new ArgumentTypeException("x") : null);
            Utils.ThrowException((y != null && !(y is T)) ? new ArgumentTypeException("y") : null);
            return Equals((T)x, (T)y);
        }
        int IEqualityComparer.GetHashCode(object obj)
        {
            Utils.ThrowException((obj != null && !(obj is T)) ? new ArgumentTypeException("obj") : null);
            return GetHashCode((T)obj); // throws ArgumentNullException
        }
    }

    /* .-----------------------------------------------------------------------
       |
       |  Class SetEqualityComparer<T>
       |
       '-----------------------------------------------------------------------
    */
    public class SetEqualityComparer<T> : IEqualityComparer<Set<T>>, IEqualityComparer where T : IComparable
    {
        public bool Equals(Set<T> x, Set<T> y)
        {
            if (x == null && y == null) { return true; }
            if (x == null || y == null) { return false; }
            ArrayList<T> aux_1 = new ArrayList<T>(x);
            aux_1.Sort();
            ArrayList<T> aux_2 = new ArrayList<T>(y);
            aux_2.Sort();
            return aux_1.ContentEquals(aux_2);
        }
        public int GetHashCode(Set<T> obj)
        {
            ArrayList<T> aux = new ArrayList<T>(obj); // throws ArgumentNullException
            aux.Sort();
            return Utils.GetHashCode(aux);
        }
        bool IEqualityComparer.Equals(object x, object y)
        {
            Utils.ThrowException((x != null && !(x is Set<T>)) ? new ArgumentTypeException("x") : null);
            Utils.ThrowException((y != null && !(y is Set<T>)) ? new ArgumentTypeException("y") : null);
            return Equals((Set<T>)x, (Set<T>)y);
        }
        int IEqualityComparer.GetHashCode(object obj)
        {
            Utils.ThrowException((obj != null && !(obj is Set<T>)) ? new ArgumentTypeException("obj") : null);
            return GetHashCode((Set<T>)obj); // throws ArgumentNullException
        }
    }

    /* .-----------------------------------------------------------------------
       |
       |  Class DescSort<T>
       |
       '-----------------------------------------------------------------------
    */
    public class DescSort<T> : IComparer<T>
    {
        public int Compare(T x, T y)
        {
            if (x == null && y == null) { return 0; }
            else if (x == null) { return -1; }
            else if (y == null) { return 1; }
            else { return ((IComparable<T>)y).CompareTo(x); }
        }
    }

    /* .-----------------------------------------------------------------------
       |
       |  Class ListEnum
       |
       '-----------------------------------------------------------------------
    */
    public class ListEnum : IEnumerator
    {
        private IEnumerableList m_list;
        private int m_current_idx
            = -1;
        public ListEnum(IEnumerableList list)
        {
            Utils.ThrowException(list == null ? new ArgumentNullException("list") : null);
            m_list = list;
        }
        // *** IEnumerator interface implementation ***
        public void Reset()
        {
            m_current_idx = -1;
        }
        public object Current
        {
            get { return m_list[m_current_idx]; }
        }
        public bool MoveNext()
        {
            m_current_idx++;
            if (m_current_idx >= m_list.Count) { Reset(); return false; }
            return true;
        }
    }

    /* .-----------------------------------------------------------------------
       |
       |  Class ListEnum<T>
       |
       '-----------------------------------------------------------------------
    */
    public class ListEnum<T> : IEnumerator<T>
    {
        private IEnumerableList<T> m_list;
        private int m_current_idx
            = -1;
        public ListEnum(IEnumerableList<T> list)
        {
            Utils.ThrowException(list == null ? new ArgumentNullException("list") : null);
            m_list = list;
        }
        // *** IEnumerator<T> interface implementation ***
        public void Reset()
        {
            m_current_idx = -1;
        }
        public T Current
        {
            get { return m_list[m_current_idx]; }
        }
        object IEnumerator.Current
        {
            get { return Current; }
        }
        public bool MoveNext()
        {
            m_current_idx++;
            if (m_current_idx >= m_list.Count) { Reset(); return false; }
            return true;
        }
        public void Dispose()
        {
        }
    }
}