/*==========================================================================;
 *
 *  (c) 2007-08 JSI.  All rights reserved.
 *
 *  File:          SparseVector2.cs
 *  Version:       1.0
 *  Desc:		   Sparse vector data structure 
 *  Author:        Miha Grcar
 *  Created on:    Mar-2007
 *  Last modified: Jul-2008
 *  Revision:      Jul-2008
 *
 ***************************************************************************/

using System;
using System.Collections;
using System.Collections.Generic;
using System.Text;
//using Latino.TextGarden;

namespace Latino
{
    /* .-----------------------------------------------------------------------
       |
       |  Class SparseVector2<T>
       |
       '-----------------------------------------------------------------------
    */
    public class SparseVector2<T> : IList, ICollection, IEnumerable<IdxDat2<T>>, ICloneable<SparseVector2<T>>, IDeeplyCloneable<SparseVector2<T>>,
        IContentEquatable<SparseVector2<T>>, ISerializable
    {
        private ArrayList<IdxDat2<T>> m_items
            = new ArrayList<IdxDat2<T>>();
        public SparseVector2()
        {
        }
        public SparseVector2(int capacity)
        {
            m_items = new ArrayList<IdxDat2<T>>(capacity); // throws ArgumentOutOfRangeException
        }
        public SparseVector2(BinarySerializer reader)
        {
            Load(reader); // throws ArgumentNullException, serialization-related exceptions
        }
        internal ArrayList<IdxDat2<T>> Inner
        {
            get { return m_items; }
        }
        public bool ContainsAt(int index)
        {
            Utils.ThrowException(index < 0 ? new ArgumentOutOfRangeException("index") : null);
            return m_items.BinarySearch(new IdxDat2<T>(index)) >= 0;
        }
        public int FirstNonEmptyIndex
        {
            get
            {
                if (m_items.Count == 0) { return -1; }
                return m_items[0].Idx;
            }
        }
        public int LastNonEmptyIndex
        {
            get
            {
                if (m_items.Count == 0) { return -1; }
                return m_items[m_items.Count - 1].Idx;
            }
        }
        public IdxDat2<T> First
        {
            get { return m_items[0]; } // throws ArgumentOutOfRangeException
        }
        public IdxDat2<T> Last
        {
            get { return m_items[m_items.Count - 1]; } // throws ArgumentOutOfRangeException
        }
        public override string ToString()
        {
            StringBuilder str_bld = new StringBuilder("{");
            foreach (IdxDat2<T> item_info in m_items)
            {
                str_bld.Append(" ");
                str_bld.Append(item_info);
            }
            str_bld.Append(" }");
            return str_bld.ToString();
        }
        private int GetNextIdx(int current_idx)
        {
            int idx = m_items.BinarySearch(new IdxDat2<T>(current_idx + 1));
            if (idx < 0) { idx = ~idx; }
            if (idx == m_items.Count) { return -1; }
            return m_items[idx].Idx;
        }
        public static int GetNextIdxAny(SparseVector2<T>.ReadOnly vec_1, SparseVector2<T>.ReadOnly vec_2, int current_idx)
        {
            Utils.ThrowException(vec_1 == null ? new ArgumentNullException("vec_1") : null);
            Utils.ThrowException(vec_2 == null ? new ArgumentNullException("vec_2") : null);
            Utils.ThrowException(current_idx < -1 ? new ArgumentOutOfRangeException("current_idx") : null); // *** note that -1 is allowed
            int idx_1 = vec_1.Inner.GetNextIdx(current_idx);
            int idx_2 = vec_2.Inner.GetNextIdx(current_idx);
            if (idx_1 < 0) { return idx_2; }
            else if (idx_2 < 0) { return idx_1; }
            else { return idx_1 < idx_2 ? idx_1 : idx_2; }
        }
        public void Append(SparseVector2<T>.ReadOnly other_vec, int this_vec_len)
        {
            Utils.ThrowException(other_vec == null ? new ArgumentNullException("other_vec") : null);
            Utils.ThrowException(this_vec_len < 0 ? new ArgumentOutOfRangeException("this_vec_len") : null);
            foreach (IdxDat2<T> item_info in other_vec)
            {
                this[item_info.Idx + this_vec_len] = item_info.Dat; // *** note that the elements are not cloned (you need to clone them yourself if needed)
            }
        }
        public void Merge(SparseVector2<T>.ReadOnly other_vec, IBinaryOperator binary_operator)
        {
            Utils.ThrowException(other_vec == null ? new ArgumentNullException("other_vec") : null);
            Utils.ThrowException(binary_operator == null ? new ArgumentNullException("binary_operator") : null);
            int idx = -1;
            while ((idx = SparseVector2<T>.GetNextIdxAny(this, other_vec, idx)) >= 0)
            {
                this[idx] = binary_operator.PerformOperation(this[idx], other_vec[idx]);
            }
        }
        public void PerformUnaryOperation(IUnaryOperator unary_operator)
        {
            Utils.ThrowException(unary_operator == null ? new ArgumentNullException("unary_operator") : null);
            for (int i = m_items.Count - 1; i >= 0; i--)
            {
                SetDirect(i, unary_operator.PerformOperation(m_items[i].Dat));
            }
        }
        // *** Direct access ***
        public IdxDat2<T> GetDirect(int direct_idx)
        {
            return m_items[direct_idx]; // throws ArgumentOutOfRangeException
        }
        public void SetDirect(int direct_idx, object value)
        {
            Utils.ThrowException((value != null && !(value is T)) ? new ArgumentTypeException("value") : null);
            if (value == null) { m_items.RemoveAt(direct_idx); } // throws ArgumentOutOfRangeException
            else { m_items[direct_idx] = new IdxDat2<T>(m_items[direct_idx].Idx, (T)value); } // throws ArgumentOutOfRangeException
        }
        public void RemoveDirect(int direct_idx)
        {
            m_items.RemoveAt(direct_idx); // throws ArgumentOutOfRangeException
        }
        public int GetDirectIdx(int index)
        {
            int direct_idx = m_items.BinarySearch(new IdxDat2<T>(index));
            return direct_idx;
        }
        // *** Partial IList<T> interface implementation ***
        public int IndexOf(T item)
        {
            Utils.ThrowException(item == null ? new ArgumentNullException("item") : null);
            foreach (IdxDat2<T> item_info in m_items)
            {
                if (item_info.Dat.Equals(item)) { return item_info.Idx; }
            }
            return -1;
        }
        public void RemoveAt(int index)
        {
            Utils.ThrowException(index < 0 ? new ArgumentOutOfRangeException("index") : null);
            int idx = m_items.BinarySearch(new IdxDat2<T>(index));
            if (idx >= 0) { m_items.RemoveAt(idx); }
        }
        public void PurgeAt(int index)
        {
            Utils.ThrowException(index < 0 ? new ArgumentOutOfRangeException("index") : null);
            int idx = m_items.BinarySearch(new IdxDat2<T>(index));
            if (idx >= 0) { m_items.RemoveAt(idx); } else { idx = ~idx; }
            for (int i = idx; i < m_items.Count; i++) { IdxDat2<T> tmp = m_items[i]; m_items[i] = new IdxDat2<T>(tmp.Idx - 1, tmp.Dat); }
        }
        // *** IList interface implementation ***
        int IList.Add(object item)
        {
            throw new NotSupportedException();
        }
        bool IList.Contains(object item)
        {
            Utils.ThrowException(!(item is T) ? new ArgumentTypeException("item") : null);
            return Contains((T)item);
        }
        int IList.IndexOf(object item)
        {
            Utils.ThrowException(!(item is T) ? new ArgumentTypeException("item") : null);
            return IndexOf((T)item);
        }
        void IList.Insert(int index, object item)
        {
            this[index] = item; // throws ArgumentOutOfRangeException, ArgumentTypeException
        }
        public bool IsFixedSize
        {
            get { return false; }
        }
        void IList.Remove(object item)
        {
            Utils.ThrowException(!(item is T) ? new ArgumentTypeException("item") : null);
            Remove((T)item);
        }
        public object this[int index]
        {
            get
            {
                Utils.ThrowException(index < 0 ? new ArgumentOutOfRangeException("index") : null);
                int idx = m_items.BinarySearch(new IdxDat2<T>(index));
                if (idx < 0) { return null; }
                return m_items[idx].Dat;
            }
            set
            {
                Utils.ThrowException(index < 0 ? new ArgumentOutOfRangeException("index") : null);
                Utils.ThrowException((value != null && !(value is T)) ? new ArgumentTypeException("Item setter value") : null);
                if (value == null)
                {
                    RemoveAt(index);
                }
                else
                {
                    int idx = m_items.BinarySearch(new IdxDat2<T>(index));
                    if (idx >= 0)
                    {
                        m_items[idx] = new IdxDat2<T>(m_items[idx].Idx, (T)value);
                    }
                    else
                    {
                        m_items.Insert(~idx, new IdxDat2<T>(index, (T)value));
                    }
                }
            }
        }
        // *** Partial ICollection<T> interface implementation ***
        public void Clear()
        {
            m_items.Clear();
        }
        public bool Contains(T item)
        {
            return IndexOf(item) >= 0; // throws ArgumentNullException
        }
        public int Count
        {
            get { return m_items.Count; }
        }
        public bool IsReadOnly
        {
            get { return false; }
        }
        public bool Remove(T item)
        {
            int index = IndexOf(item); // throws ArgumentNullException
            if (index >= 0)
            {
                RemoveAt(index);
                return true;
            }
            return false;
        }
        // *** ICollection interface implementation ***
        void ICollection.CopyTo(Array array, int index)
        {
            throw new NotSupportedException();
        }
        bool ICollection.IsSynchronized
        {
            get { throw new NotSupportedException(); }
        }
        object ICollection.SyncRoot
        {
            get { throw new NotSupportedException(); }
        }
        // *** IEnumerable<IdxDat2<T>> interface implementation ***
        public IEnumerator<IdxDat2<T>> GetEnumerator()
        {
            return m_items.GetEnumerator();
        }
        // *** IEnumerable interface implementation ***
        IEnumerator IEnumerable.GetEnumerator()
        {
            return GetEnumerator();
        }
        // *** ICloneable interface implementation ***
        public SparseVector2<T> Clone()
        {
            SparseVector2<T> clone = new SparseVector2<T>();
            clone.m_items = m_items.Clone();
            return clone;
        }
        object ICloneable.Clone()
        {
            return Clone();
        }
        // *** IDeeplyCloneable interface implementation ***
        public SparseVector2<T> DeeplyClone()
        {
            SparseVector2<T> clone = new SparseVector2<T>();
            clone.m_items.Capacity = m_items.Capacity;
            foreach (IdxDat2<T> item_info in m_items)
            {
                clone.m_items.Add(new IdxDat2<T>(item_info.Idx, (T)Utils.Clone(item_info.Dat, /*deep_clone=*/true)));
            }
            return clone;
        }
        object IDeeplyCloneable.DeeplyClone()
        {
            return DeeplyClone();
        }
        // *** IContentEquatable<SparseVector2<T>> interface implementation ***
        public bool ContentEquals(SparseVector2<T> other)
        {
            if (other == null || m_items.Count != other.m_items.Count) { return false; }
            foreach (IdxDat2<T> item_info in m_items)
            {
                if (!Utils.ObjectEquals(item_info.Dat, other[item_info.Idx], /*deep_cmp=*/true)) { return false; }
            }
            return true;
        }
        bool IContentEquatable.ContentEquals(object other)
        {
            Utils.ThrowException((other != null && !(other is SparseVector2<T>)) ? new ArgumentTypeException("other") : null);
            return ContentEquals((SparseVector2<T>)other);
        }
        // *** ISerializable interface implementation ***
        public void Save(BinarySerializer writer)
        {
            Utils.ThrowException(writer == null ? new ArgumentNullException("writer") : null);
            // the following statements throw serialization-related exception
            writer.WriteInt(m_items.Count);
            foreach (IdxDat2<T> item_info in m_items)
            {
                writer.WriteInt(item_info.Idx);
                writer.WriteValueOrObject<T>(item_info.Dat);
            }
        }
        public void Load(BinarySerializer reader)
        {
            Utils.ThrowException(reader == null ? new ArgumentNullException("reader") : null);
            m_items.Clear();
            // the following statements throw serialization-related exception
            int count = reader.ReadInt();
            for (int i = 0; i < count; i++)
            {
                m_items.Add(new IdxDat2<T>(reader.ReadInt(), reader.ReadValueOrObject<T>()));
            }
        }
        // *** Implicit cast to a read-only adapter ***
        public static implicit operator SparseVector2<T>.ReadOnly(SparseVector2<T> vec)
        {
            if (vec == null) { return null; }
            return new SparseVector2<T>.ReadOnly(vec);
        }
        // *** Equality comparer ***
        public static IEqualityComparer<SparseVector2<T>> GetEqualityComparer()
        {
            return new GenericEqualityComparer<SparseVector2<T>>();
        }
        /* .-----------------------------------------------------------------------
           |
           |  Class SparseVector2<T>.ReadOnly
           |
           '-----------------------------------------------------------------------
        */
        public class ReadOnly : IReadOnlyAdapter<SparseVector2<T>>, ICollection, IEnumerable<IdxDat2<T>>, IContentEquatable<SparseVector2<T>.ReadOnly>,
            ISerializable
        {
            private SparseVector2<T> m_vec;
            public ReadOnly(SparseVector2<T> vec)
            {
                Utils.ThrowException(vec == null ? new ArgumentNullException("vec") : null);
                m_vec = vec;
            }
            public bool ContainsAt(int index)
            {
                return m_vec.ContainsAt(index);
            }
            public int FirstNonEmptyIndex
            {
                get { return m_vec.FirstNonEmptyIndex; }
            }
            public int LastNonEmptyIndex
            {
                get { return m_vec.LastNonEmptyIndex; }
            }
            public IdxDat2<T> First
            {
                get { return m_vec.First; }
            }
            public IdxDat2<T> Last
            {
                get { return m_vec.Last; }
            }
            public override string ToString()
            {
                return m_vec.ToString();
            }
            // *** Direct access ***
            public IdxDat2<T> GetDirect(int direct_idx)
            {
                return m_vec.GetDirect(direct_idx);
            }
            public int GetDirectIdx(int index)
            {
                return m_vec.GetDirectIdx(index);
            }
            // *** IReadOnlyAdapter interface implementation ***
            public SparseVector2<T> GetWritableCopy()
            {
                return m_vec.Clone();
            }
            object IReadOnlyAdapter.GetWritableCopy()
            {
                return GetWritableCopy();
            }
            internal SparseVector2<T> Inner
            {
                get { return m_vec; }
            }
            // *** Partial IList<T> interface implementation ***
            public int IndexOf(T item)
            {
                return m_vec.IndexOf(item);
            }
            // *** Partial IList interface implementation ***
            public object this[int index]
            {
                get { return m_vec[index]; }
            }
            // *** Partial ICollection<T> interface implementation ***
            public bool Contains(T item)
            {
                return m_vec.Contains(item);
            }
            public int Count
            {
                get { return m_vec.Count; }
            }
            public bool IsReadOnly
            {
                get { return true; }
            }
            // *** ICollection interface implementation ***
            void ICollection.CopyTo(Array array, int index)
            {
                throw new NotSupportedException();
            }
            bool ICollection.IsSynchronized
            {
                get { throw new NotSupportedException(); }
            }
            object ICollection.SyncRoot
            {
                get { throw new NotSupportedException(); }
            }
            // *** IEnumerable<IdxDat2<T>> interface implementation ***
            public IEnumerator<IdxDat2<T>> GetEnumerator()
            {
                return m_vec.GetEnumerator();
            }
            // *** IEnumerable interface implementation ***
            IEnumerator IEnumerable.GetEnumerator()
            {
                return ((IEnumerable)m_vec).GetEnumerator();
            }
            // *** IContentEquatable<SparseVector2<T>.ReadOnly> interface implementation ***
            public bool ContentEquals(SparseVector2<T>.ReadOnly other)
            {
                return other != null && m_vec.ContentEquals(other.Inner);
            }
            bool IContentEquatable.ContentEquals(object other)
            {
                Utils.ThrowException((other != null && !(other is SparseVector2<T>.ReadOnly)) ? new ArgumentTypeException("other") : null);
                return ContentEquals((SparseVector2<T>.ReadOnly)other);
            }
            // *** ISerializable interface implementation ***
            public void Save(BinarySerializer writer)
            {
                m_vec.Save(writer);
            }
            // *** Equality comparer ***
            public static IEqualityComparer<SparseVector2<T>.ReadOnly> GetEqualityComparer()
            {
                return new GenericEqualityComparer<SparseVector2<T>.ReadOnly>();
            }
        }
    }
}
