/*==========================================================================;
 *
 *  (c) 2007-08 JSI.  All rights reserved.
 *
 *  File:          SparseMatrix2.cs
 *  Version:       1.0
 *  Desc:		   Sparse matrix data structure 
 *  Author:        Miha Grcar
 *  Created on:    Apr-2007
 *  Last modified: Jul-2008
 *  Revision:      Jul-2008
 *
 ***************************************************************************/

using System;
using System.Collections;
using System.Collections.Generic;
using System.Text;

namespace Latino
{
    /* .-----------------------------------------------------------------------
       |
       |  Class SparseMatrix2<T>
       |
       '-----------------------------------------------------------------------
    */
    public class SparseMatrix2<T> : IEnumerable<IdxDat2<SparseVector2<T>>>, ICloneable<SparseMatrix2<T>>, IDeeplyCloneable<SparseMatrix2<T>>, IContentEquatable<SparseMatrix2<T>>,
        ISerializable
    {
        private SparseVector2<SparseVector2<T>> m_rows
            = new SparseVector2<SparseVector2<T>>();
        public SparseMatrix2()
        {
        }
        public SparseMatrix2(BinarySerializer reader)
        {
            Load(reader); // throws ArgumentNullException
        }
        public bool ContainsRowAt(int row_idx)
        {
            return m_rows[row_idx] != null; // throws ArgumentOutOfRangeException
        }
        public bool ContainsColAt(int col_idx)
        {
            Utils.ThrowException(col_idx < 0 ? new ArgumentOutOfRangeException("col_idx") : null);
            foreach (IdxDat2<SparseVector2<T>> row in m_rows)
            {
                if (row.Dat.ContainsAt(col_idx)) { return true; }
            }
            return false;
        }
        public bool ContainsAt(int row_idx, int col_idx)
        {
            Utils.ThrowException(row_idx < 0 ? new ArgumentOutOfRangeException("row_idx") : null);
            Utils.ThrowException(col_idx < 0 ? new ArgumentOutOfRangeException("col_idx") : null);
            SparseVector2<T> row = (SparseVector2<T>)m_rows[row_idx];
            if (row == null) { return false; }
            return row.ContainsAt(col_idx);
        }
        public int FirstNonEmptyRowIdx
        {
            get { return m_rows.FirstNonEmptyIndex; }
        }
        public int LastNonEmptyRowIdx
        {
            get { return m_rows.LastNonEmptyIndex; }
        }
        public int GetFirstNonEmptyColIdx()
        {
            int min_idx = int.MaxValue;
            foreach (IdxDat2<SparseVector2<T>> row in m_rows)
            {
                if (row.Dat.FirstNonEmptyIndex < min_idx)
                {
                    min_idx = row.Dat.FirstNonEmptyIndex;
                }
            }
            return min_idx == int.MaxValue ? -1 : min_idx;
        }
        public int GetLastNonEmptyColIdx()
        {
            int max_idx = -1;
            foreach (IdxDat2<SparseVector2<T>> row in m_rows)
            {
                if (row.Dat.LastNonEmptyIndex > max_idx)
                {
                    max_idx = row.Dat.LastNonEmptyIndex;
                }
            }
            return max_idx;
        }
        public override string ToString()
        {
            return m_rows.ToString();
        }
        public string ToString(string format)
        {
            if (format == "C") // compact format
            {
                return m_rows.ToString();
            }
            else if (format == "E") // extended format
            {
                StringBuilder str_bld = new StringBuilder();
                int row_count = LastNonEmptyRowIdx + 1;
                int col_count = GetLastNonEmptyColIdx() + 1;
                for (int row_idx = 0; row_idx < row_count; row_idx++)
                {
                    for (int col_idx = 0; col_idx < col_count; col_idx++)
                    {
                        object val = this[row_idx, col_idx];
                        str_bld.Append(val == null ? "-" : val.ToString());
                        if (col_idx != col_count - 1) { str_bld.Append("\t"); }
                    }
                    if (row_idx != row_count - 1) { str_bld.Append("\n"); }
                }
                return str_bld.ToString();
            }
            else
            {
                throw new ArgumentNotSupportedException("format");
            }
        }
        public bool IndexOf(T val, ref int row_idx, ref int col_idx)
        {
            Utils.ThrowException(val != null ? new ArgumentNullException("val") : null);
            foreach (IdxDat2<SparseVector2<T>> row in m_rows)
            {
                foreach (IdxDat2<T> item in row.Dat)
                {
                    if (item.Dat.Equals(val))
                    {
                        row_idx = row.Idx;
                        col_idx = item.Idx;
                        return true;
                    }
                }
            }
            return false;
        }
        public bool Contains(T val)
        {
            int foo = 0, bar = 0;
            return IndexOf(val, ref foo, ref bar); // throws ArgumentNullException
        }
        public bool IsReadOnly
        {
            get { return false; }
        }
        public int RowCount
        {
            get { return m_rows.Count; }
        }
        public int CountValues()
        {
            int count = 0;
            foreach (IdxDat2<SparseVector2<T>> row_data in m_rows)
            {
                count += row_data.Dat.Count;
            }
            return count;
        }
        public double GetSparseness(int num_rows, int num_cols)
        {
            int all_values = num_rows * num_cols;
            return (double)(all_values - CountValues()) / (double)all_values;
        }
        public bool ContainsEmptyRows()
        {
            foreach (IdxDat2<SparseVector2<T>> row_data in m_rows)
            {
                if (row_data.Dat.Count == 0) { return true; }
            }
            return false;
        }
        public SparseVector2<T> this[int row_idx] // *** produces empty rows
        {
            get { return (SparseVector2<T>)m_rows[row_idx]; } // throws ArgumentOutOfRangeException
            set { m_rows[row_idx] = value; } // throws ArgumentOutOfRangeException
        }
        public object this[int row_idx, int col_idx]
        {
            get
            {
                Utils.ThrowException(row_idx < 0 ? new ArgumentOutOfRangeException("row_idx") : null);
                Utils.ThrowException(col_idx < 0 ? new ArgumentOutOfRangeException("col_idx") : null);
                SparseVector2<T> row = (SparseVector2<T>)m_rows[row_idx];
                if (row == null) { return null; }
                return row[col_idx];
            }
            set
            {
                Utils.ThrowException((value != null && !(value is T)) ? new ArgumentTypeException("Item setter value") : null);
                Utils.ThrowException(row_idx < 0 ? new ArgumentOutOfRangeException("row_idx") : null);
                Utils.ThrowException(col_idx < 0 ? new ArgumentOutOfRangeException("col_idx") : null);
                if (value == null)
                {
                    RemoveAt(row_idx, col_idx);
                }
                else
                {
                    SparseVector2<T> row = (SparseVector2<T>)m_rows[row_idx];
                    if (row == null) { m_rows[row_idx] = row = new SparseVector2<T>(); }
                    row[col_idx] = value;
                }
            }
        }
        public void SetDiagonal(object val, int matrix_dim)
        {
            Utils.ThrowException((val != null && !(val is T)) ? new ArgumentTypeException("val") : null);
            Utils.ThrowException(matrix_dim < 0 ? new ArgumentOutOfRangeException("matrix_dim") : null);
            for (int i = 0; i < matrix_dim; i++)
            {
                this[i, i] = val;
            }
        }
        public SparseVector2<T> GetColCopy(int col_idx)
        {
            Utils.ThrowException(col_idx < 0 ? new ArgumentOutOfRangeException("col_idx") : null);
            SparseVector2<T> col = new SparseVector2<T>();
            foreach (IdxDat2<SparseVector2<T>> row in m_rows)
            {
                object val = row.Dat[col_idx];
                if (val != null) { col[row.Idx] = val; }
            }
            return col;
        }
        public SparseMatrix2<T> GetTransposedCopy()
        {
            SparseMatrix2<T> tr_mat = new SparseMatrix2<T>();
            foreach (IdxDat2<SparseVector2<T>> row_data in m_rows)
            {
                foreach (IdxDat2<T> item_data in row_data.Dat)
                {
                    tr_mat[item_data.Idx, row_data.Idx] = item_data.Dat;
                }
            }
            return tr_mat;
        }
        public bool Remove(T val)
        {
            Utils.ThrowException(val == null ? new ArgumentNullException("val") : null);
            bool val_found = false;
            for (int direct_row_idx = m_rows.Count - 1; direct_row_idx >= 0; direct_row_idx--)
            {
                SparseVector2<T> row = m_rows.GetDirect(direct_row_idx).Dat;
                for (int direct_col_idx = row.Count - 1; direct_col_idx >= 0; direct_col_idx--)
                {
                    if (val.Equals(row.GetDirect(direct_col_idx).Dat))
                    {
                        row.RemoveDirect(direct_col_idx);
                        if (row.Count == 0) { RemoveRowDirect(direct_row_idx); }
                        val_found = true;
                    }
                }
            }
            return val_found;
        }
        public void RemoveAt(int row_idx, int col_idx)
        {
            Utils.ThrowException(row_idx < 0 ? new ArgumentOutOfRangeException("row_idx") : null);
            Utils.ThrowException(col_idx < 0 ? new ArgumentOutOfRangeException("col_idx") : null);
            int direct_row_idx = m_rows.GetDirectIdx(row_idx);
            if (direct_row_idx >= 0)
            {
                SparseVector2<T> row = m_rows.GetDirect(direct_row_idx).Dat;
                row.RemoveAt(col_idx);
                if (row.Count == 0)
                {
                    m_rows.RemoveDirect(direct_row_idx);
                }
            }
        }
        public void RemoveRowAt(int row_idx)
        {
            m_rows.RemoveAt(row_idx); // throws ArgumentOutOfRangeException
        }
        public void RemoveColAt(int col_idx)
        {
            Utils.ThrowException(col_idx < 0 ? new ArgumentOutOfRangeException("col_idx") : null);
            for (int i = m_rows.Count - 1; i >= 0; i--)
            {
                SparseVector2<T> row = m_rows.GetDirect(i).Dat;
                row.RemoveAt(col_idx);
                if (row.Count == 0)
                {
                    m_rows.RemoveDirect(i);
                }
            }
        }
        public void PurgeColAt(int col_idx)
        {
            Utils.ThrowException(col_idx < 0 ? new ArgumentOutOfRangeException("col_idx") : null);
            for (int i = m_rows.Count - 1; i >= 0; i--)
            {
                SparseVector2<T> row = m_rows.GetDirect(i).Dat;
                row.PurgeAt(col_idx);
                if (row.Count == 0)
                {
                    m_rows.RemoveDirect(i);
                }
            }
        }
        public void PurgeRowAt(int row_idx)
        {
            m_rows.PurgeAt(row_idx); // throws ArgumentOutOfRangeException
        }
        public void RemoveEmptyRows()
        {
            for (int row_idx = m_rows.Count - 1; row_idx >= 0; row_idx--)
            {
                if (m_rows.GetDirect(row_idx).Dat.Count == 0)
                {
                    m_rows.RemoveDirect(row_idx);
                }
            }
        }
        public void PurgeEmptyRows()
        {
            SparseVector2<SparseVector2<T>> new_rows = new SparseVector2<SparseVector2<T>>();
            int row_idx = 0;
            foreach (IdxDat2<SparseVector2<T>> row in m_rows)
            {
                if (row.Dat.Count > 0)
                {
                    new_rows[row_idx++] = row.Dat;
                }
            }
            m_rows = new_rows;
        }
        public void Clear()
        {
            m_rows.Clear();
        }
        public void AppendCols(SparseMatrix2<T>.ReadOnly other_matrix, int this_matrix_num_cols) // *** note that the elements are not cloned (you need to clone them yourself if needed)
        {
            Utils.ThrowException(other_matrix == null ? new ArgumentNullException("other_matrix") : null);
            Utils.ThrowException(this_matrix_num_cols < 0 ? new ArgumentOutOfRangeException("this_matrix_num_cols") : null);
            int idx = -1;
            SparseMatrix2<T> other_matrix_inner = other_matrix.Inner;
            while ((idx = SparseVector2<SparseVector2<T>>.GetNextIdxAny(m_rows, other_matrix_inner.m_rows, idx)) >= 0)
            {
                int direct_idx = m_rows.GetDirectIdx(idx);
                if (direct_idx < 0)
                {
                    SparseVector2<T> new_row = new SparseVector2<T>();
                    new_row.Append(other_matrix_inner[idx], this_matrix_num_cols);
                    this[idx] = new_row;
                }
                else
                {
                    SparseVector2<T> row = other_matrix_inner[idx];
                    if (row != null)
                    {
                        GetRowDirect(direct_idx).Dat.Append(row, this_matrix_num_cols);
                    }
                }
            }
        }
        public void Merge(SparseMatrix2<T>.ReadOnly other_matrix, IBinaryOperator binary_operator)
        {
            Utils.ThrowException(other_matrix == null ? new ArgumentNullException("other_matrix") : null);
            Utils.ThrowException(binary_operator == null ? new ArgumentNullException("binary_operator") : null);
            int idx = -1;
            SparseMatrix2<T> other_matrix_inner = other_matrix.Inner;
            while ((idx = SparseVector2<SparseVector2<T>>.GetNextIdxAny(m_rows, other_matrix_inner.m_rows, idx)) >= 0)
            {
                int direct_idx = m_rows.GetDirectIdx(idx);
                if (direct_idx < 0)
                {
                    SparseVector2<T> new_row = new SparseVector2<T>();
                    new_row.Merge(other_matrix_inner[idx], binary_operator);
                    if (new_row.Count > 0) { this[idx] = new_row; }
                }
                else
                {
                    SparseVector2<T> other_row = other_matrix_inner[idx];
                    if (other_row != null)
                    {
                        SparseVector2<T> row = GetRowDirect(direct_idx).Dat;
                        row.Merge(other_row, binary_operator);
                        if (row.Count == 0) { RemoveRowDirect(direct_idx); }
                    }
                }
            }
        }
        public void PerformUnaryOperation(IUnaryOperator unary_operator)
        {
            Utils.ThrowException(unary_operator == null ? new ArgumentNullException("unary_operator") : null);
            for (int row_idx = m_rows.Count - 1; row_idx >= 0; row_idx--)
            {
                SparseVector2<T> row = m_rows.GetDirect(row_idx).Dat;
                row.PerformUnaryOperation(unary_operator);
                if (row.Count == 0)
                {
                    m_rows.RemoveDirect(row_idx);
                }
            }
        }
        // *** Direct access ***
        public IdxDat2<SparseVector2<T>> GetRowDirect(int direct_idx) // *** produces empty rows
        {
            return m_rows.GetDirect(direct_idx);
        }
        public void SetRowDirect(int direct_idx, SparseVector2<T> row) 
        {
            if (row == null || row.Count == 0)
            {
                m_rows.RemoveDirect(direct_idx); // throws ArgumentOutOfRangeException
            }
            else
            {
                m_rows.SetDirect(direct_idx, row); // throws ArgumentOutOfRangeException
            }
        }
        public void RemoveRowDirect(int direct_idx)
        {
            m_rows.RemoveDirect(direct_idx); // throws ArgumentOutOfRangeException
        }
        public T GetDirect(int direct_row_idx, int direct_col_idx)
        {
            return m_rows.GetDirect(direct_row_idx).Dat.GetDirect(direct_col_idx).Dat; // throws ArgumentOutOfRangeException
        }
        public void SetDirect(int direct_row_idx, int direct_col_idx, object item)
        {
            Utils.ThrowException((item != null && !(item is T)) ? new ArgumentTypeException("item") : null);
            SparseVector2<T> row = m_rows.GetDirect(direct_row_idx).Dat; // throws ArgumentOutOfRangeException
            row.SetDirect(direct_col_idx, item); // throws ArgumentOutOfRangeException
            if (row.Count == 0)
            {
                m_rows.RemoveDirect(direct_row_idx);
            }
        }
        public void RemoveDirect(int direct_row_idx, int direct_col_idx)
        {
            SparseVector2<T> row = m_rows.GetDirect(direct_row_idx).Dat; // throws ArgumentOutOfRangeException
            row.RemoveDirect(direct_col_idx); // throws ArgumentOutOfRangeException
            if (row.Count == 0)
            {
                m_rows.RemoveDirect(direct_row_idx);
            }
        }
        public int GetDirectRowIdx(int row_idx)
        {
            return m_rows.GetDirectIdx(row_idx);
        }
        // *** IEnumerable<IdxDat2<SparseVector2<T>>> interface implementation ***
        public IEnumerator<IdxDat2<SparseVector2<T>>> GetEnumerator() // *** produces empty rows
        {
            return m_rows.GetEnumerator();
        }
        // *** IEnumerable interface implementation ***
        IEnumerator IEnumerable.GetEnumerator() // *** produces empty rows
        {
            return GetEnumerator();
        }
        // *** ICloneable interface implementation ***
        public SparseMatrix2<T> Clone()
        {
            SparseMatrix2<T> clone = new SparseMatrix2<T>();
            clone.m_rows = m_rows.Clone();
            int i = 0;
            foreach (IdxDat2<SparseVector2<T>> row in m_rows)
            {
                clone.m_rows.SetDirect(i++, row.Dat.Clone());
            }
            return clone;
        }
        object ICloneable.Clone()
        {
            return Clone();
        }
        // *** IDeeplyCloneable interface implementation ***
        public SparseMatrix2<T> DeeplyClone()
        {
            SparseMatrix2<T> clone = new SparseMatrix2<T>();
            clone.m_rows = m_rows.DeeplyClone();
            return clone;
        }
        object IDeeplyCloneable.DeeplyClone()
        {
            return DeeplyClone();
        }
        // *** IContentEquatable<SparseMatrix2<T>> interface implementation ***
        public bool ContentEquals(SparseMatrix2<T> other)
        {
            return other != null && m_rows.ContentEquals(other.m_rows);
        }
        bool IContentEquatable.ContentEquals(object other)
        {
            Utils.ThrowException((other != null && !(other is SparseMatrix2<T>)) ? new ArgumentTypeException("other") : null);
            return ContentEquals((SparseMatrix2<T>)other);
        }
        // *** ISerializable interface implementation ***
        public void Save(BinarySerializer writer)
        {
            Utils.ThrowException(writer == null ? new ArgumentNullException("writer") : null);
            m_rows.Save(writer);
        }
        public void Load(BinarySerializer reader)
        {
            Utils.ThrowException(reader == null ? new ArgumentNullException("reader") : null);
            m_rows = new SparseVector2<SparseVector2<T>>(reader);
        }
        // *** Implicit cast to a read-only adapter ***
        public static implicit operator SparseMatrix2<T>.ReadOnly(SparseMatrix2<T> matrix)
        {
            if (matrix == null) { return null; }
            return new SparseMatrix2<T>.ReadOnly(matrix);
        }
        // *** Equality comparer ***
        // *** note that two matrices are never equal if one has empty rows and the other doesn't
        public static IEqualityComparer<SparseMatrix2<T>> GetEqualityComparer()
        {
            return new GenericEqualityComparer<SparseMatrix2<T>>();
        }
        ///* .-----------------------------------------------------------------------
        //   |
        //   |  Class ReadOnlyMatrixEnumerator
        //   |
        //   '-----------------------------------------------------------------------
        //*/
        private class ReadOnlyMatrixEnumerator : IEnumerator<IdxDat2<SparseVector2<T>.ReadOnly>>
        {
            private SparseMatrix2<T>.ReadOnly m_matrix;
            private int m_current_idx
                = -1;
            public ReadOnlyMatrixEnumerator(SparseMatrix2<T>.ReadOnly matrix)
            {
                m_matrix = matrix;
            }
            // *** IEnumerator<IdxDat2<SparseVector2<T>.ReadOnly>> interface implementation ***
            public IdxDat2<SparseVector2<T>.ReadOnly> Current
            {
                get { return m_matrix.GetRowDirect(m_current_idx); }
            }
            // *** IDisposable interface implementation ***
            public void Dispose()
            {
            }
            // *** IEnumerator interface implementation ***
            object IEnumerator.Current
            {
                get { return Current; }
            }
            public bool MoveNext()
            {
                m_current_idx++;
                if (m_current_idx >= m_matrix.RowCount) { Reset(); return false; }
                return true;
            }
            public void Reset()
            {
                m_current_idx = -1;
            }
        }
        /* .-----------------------------------------------------------------------
           |
           |  Class SparseMatrix2<T>.ReadOnly
           |
           '-----------------------------------------------------------------------
        */
        public class ReadOnly : IReadOnlyAdapter<SparseMatrix2<T>>, IEnumerable<IdxDat2<SparseVector2<T>.ReadOnly>>, IContentEquatable<SparseMatrix2<T>.ReadOnly>,
            ISerializable
        {
            private SparseMatrix2<T> m_matrix;
            public ReadOnly(SparseMatrix2<T> matrix)
            {
                Utils.ThrowException(matrix == null ? new ArgumentNullException("matrix") : null);
                m_matrix = matrix;
            }
            public bool ContainsRowAt(int row_idx)
            {
                return m_matrix.ContainsRowAt(row_idx);
            }
            public bool ContainsColAt(int col_idx)
            {
                return m_matrix.ContainsColAt(col_idx);
            }
            public bool ContainsAt(int row_idx, int col_idx)
            {
                return m_matrix.ContainsAt(row_idx, col_idx);
            }
            public int FirstNonEmptyRowIdx
            {
                get { return m_matrix.FirstNonEmptyRowIdx; }
            }
            public int LastNonEmptyRowIdx
            {
                get { return m_matrix.LastNonEmptyRowIdx; }
            }
            public int GetFirstNonEmptyColIdx()
            {
                return m_matrix.GetFirstNonEmptyColIdx();
            }
            public int GetLastNonEmptyColIdx()
            {
                return m_matrix.GetLastNonEmptyColIdx();
            }
            public override string ToString()
            {
                return m_matrix.ToString();
            }
            public string ToString(string format)
            {
                return m_matrix.ToString(format);
            }
            public bool IndexOf(T val, ref int row_idx, ref int col_idx)
            {
                return m_matrix.IndexOf(val, ref row_idx, ref col_idx);
            }
            public bool Contains(T val)
            {
                return m_matrix.Contains(val);
            }
            public bool IsReadOnly
            {
                get { return true; }
            }
            public int RowCount
            {
                get { return m_matrix.RowCount; }
            }
            public int CountValues()
            {
                return m_matrix.CountValues();
            }
            public double GetSparseness(int num_rows, int num_cols)
            {
                return m_matrix.GetSparseness(num_rows, num_cols);
            }
            public bool ContainsEmptyRows()
            {
                return m_matrix.ContainsEmptyRows();
            }
            public SparseVector2<T>.ReadOnly this[int row_idx]
            {
                get { return m_matrix[row_idx]; }
            }
            public object this[int row_idx, int col_idx]
            {
                get { return m_matrix[row_idx, col_idx]; }
            }
            public SparseVector2<T> GetColCopy(int col_idx)
            {
                return m_matrix.GetColCopy(col_idx);
            }
            public SparseMatrix2<T> GetTransposedCopy()
            {
                return m_matrix.GetTransposedCopy();
            }
            // *** Direct access ***
            public IdxDat2<SparseVector2<T>.ReadOnly> GetRowDirect(int direct_idx)
            {
                IdxDat2<SparseVector2<T>> row_data = m_matrix.GetRowDirect(direct_idx);
                return new IdxDat2<SparseVector2<T>.ReadOnly>(row_data.Idx, row_data.Dat);
            }
            public T GetDirect(int direct_row_idx, int direct_col_idx)
            {
                return m_matrix.GetDirect(direct_row_idx, direct_col_idx);
            }
            public int GetDirectRowIdx(int row_idx)
            {
                return m_matrix.GetDirectRowIdx(row_idx);
            }
            // *** IReadOnlyAdapter interface implementation ***
            public SparseMatrix2<T> GetWritableCopy()
            {
                return m_matrix.Clone();
            }
            object IReadOnlyAdapter.GetWritableCopy()
            {
                return GetWritableCopy();
            }
            internal SparseMatrix2<T> Inner
            {
                get { return m_matrix; }
            }
            // *** IEnumerable<IdxDat2<SparseVector2<T>.ReadOnly>> interface implementation ***
            public IEnumerator<IdxDat2<SparseVector2<T>.ReadOnly>> GetEnumerator()
            {
                return new ReadOnlyMatrixEnumerator(this);
            }
            // *** IEnumerable interface implementation ***
            IEnumerator IEnumerable.GetEnumerator()
            {
                return GetEnumerator();
            }
            // *** IContentEquatable<SparseMatrix2<T>.ReadOnly> interface implementation ***
            public bool ContentEquals(SparseMatrix2<T>.ReadOnly other)
            {
                return other != null && m_matrix.ContentEquals(other.Inner);
            }
            bool IContentEquatable.ContentEquals(object other)
            {
                Utils.ThrowException(other != null && !(other is SparseMatrix2<T>.ReadOnly) ? new ArgumentTypeException("other") : null);
                return ContentEquals((SparseMatrix2<T>.ReadOnly)other);
            }
            // *** ISerializable interface implementation ***
            public void Save(BinarySerializer writer)
            {
                m_matrix.Save(writer);
            }
            // *** Equality comparer ***
            // *** note that two matrices are not equal if one has empty rows and the other doesn't
            public static IEqualityComparer<SparseMatrix2<T>.ReadOnly> GetEqualityComparer()
            {
                return new GenericEqualityComparer<SparseMatrix2<T>.ReadOnly>();
            }
        }
    }
}