/*==========================================================================;
 *
 *  (c) 2007-08 JSI.  All rights reserved.
 *
 *  File:          SvmLight.cs
 *  Version:       1.0
 *  Desc:		   SVM^light and SVM^multiclass C# wrapper
 *  Author:        Miha Grcar 
 *  Created on:    Aug-2007
 *  Last modified: Oct-2008 
 *  Revision:      Oct-2008
 * 
 ***************************************************************************/

using System;
using System.Runtime.InteropServices;
using System.Collections;
using System.Collections.Generic;

namespace Latino.Model
{
    /* .-----------------------------------------------------------------------
       |		 
       |  Class SvmLightLib 
       |
       '-----------------------------------------------------------------------
    */
    internal class SvmLightLib
    {
#if DEBUG
        const string SVMLIGHTLIB_DLL = "SvmLightLibDebug.dll";
#else
        const string SVMLIGHTLIB_DLL = "SvmLightLib.dll";
#endif
        // label is 1 or -1 for inductive binary SVM; 1, -1, or 0 (unlabeled) for transductive binary SVM; 
        // a positive integer for multiclass SVM; a real value for SVM regression
        [DllImport(SVMLIGHTLIB_DLL)]
        public static extern int NewFeatureVector(int feature_count, int[] features, float[] weights, double label);
        [DllImport(SVMLIGHTLIB_DLL)]
        public static extern void DeleteFeatureVector(int id);
        [DllImport(SVMLIGHTLIB_DLL)]
        public static extern int GetFeatureVectorFeatureCount(int feature_vector_id);
        [DllImport(SVMLIGHTLIB_DLL)]
        public static extern int GetFeatureVectorFeature(int feature_vector_id, int feature_idx);
        [DllImport(SVMLIGHTLIB_DLL)]
        public static extern float GetFeatureVectorWeight(int feature_vector_id, int feature_idx);
        [DllImport(SVMLIGHTLIB_DLL)]
        public static extern double GetFeatureVectorLabel(int feature_vector_id);
        [DllImport(SVMLIGHTLIB_DLL)]
        public static extern void SetFeatureVectorLabel(int feature_vector_id, double label);
        [DllImport(SVMLIGHTLIB_DLL)]
        public static extern int GetFeatureVectorClassifScoreCount(int feature_vector_id);
        [DllImport(SVMLIGHTLIB_DLL)]
        public static extern double GetFeatureVectorClassifScore(int feature_vector_id, int classif_score_idx);

        [DllImport(SVMLIGHTLIB_DLL)]
        public static extern void _TrainModel(string args);
        [DllImport(SVMLIGHTLIB_DLL)]
        public static extern int TrainModel(string args, int feature_vector_count, int[] feature_vectors);

        [DllImport(SVMLIGHTLIB_DLL)]
        public static extern void _TrainMulticlassModel(string args);
        [DllImport(SVMLIGHTLIB_DLL)]
        public static extern int TrainMulticlassModel(string args, int feature_vector_count, int[] feature_vectors);
        [DllImport(SVMLIGHTLIB_DLL)]
        public static extern void SaveMulticlassModel(int model_id, string file_name);
        [DllImport(SVMLIGHTLIB_DLL)]
        public static extern int LoadMulticlassModel(string file_name);
        [DllImport(SVMLIGHTLIB_DLL)]
        public static extern void SaveMulticlassModelBin(int model_id, string file_name);
        [DllImport(SVMLIGHTLIB_DLL)]
        public static extern int LoadMulticlassModelBin(string file_name);

        [DllImport(SVMLIGHTLIB_DLL)]
        public static extern void _Classify(string args);
        [DllImport(SVMLIGHTLIB_DLL)]
        public static extern void Classify(int model_id, int feature_vector_count, int[] feature_vectors);

        [DllImport(SVMLIGHTLIB_DLL)]
        public static extern void _MulticlassClassify(string args);
        [DllImport(SVMLIGHTLIB_DLL)]
        public static extern void MulticlassClassify(int model_id, int feature_vector_count, int[] feature_vectors);

        [DllImport(SVMLIGHTLIB_DLL)]
        public static extern void DeleteModel(int id);
        [DllImport(SVMLIGHTLIB_DLL)]
        public static extern void DeleteMulticlassModel(int id);
    }

    /* .-----------------------------------------------------------------------
       |		 
       |  Class SvmFeatureVector 
       |
       '-----------------------------------------------------------------------
    */
    public class SvmFeatureVector : IDisposable, IEnumerableList<IdxDat2<float>>
    {
        private int m_id
            = -1;
        public SvmFeatureVector(SparseVector2<float> data)
        {
            Utils.ThrowException(data == null ? new ArgumentNullException("data") : null);
            int[] features = new int[data.Count];
            float[] weights = new float[data.Count];
            for (int i = 0; i < data.Count; i++)
            {
                features[i] = data.GetDirect(i).Idx + 1;                
                weights[i] = data.GetDirect(i).Dat;
            }
            m_id = SvmLightLib.NewFeatureVector(features.Length, features, weights, /*label=*/0);
        }
        ~SvmFeatureVector()
        {
            Dispose();
        }
        internal int Id
        {
            get { return m_id; }
        }
        public int GetFeatureId(int idx)
        {
            Utils.ThrowException(idx < 0 || idx >= Count ? new ArgumentOutOfRangeException("idx") : null);
            return SvmLightLib.GetFeatureVectorFeature(m_id, idx) - 1;
        }
        public float GetFeatureWeight(int idx)
        {
            Utils.ThrowException(idx < 0 || idx >= Count ? new ArgumentOutOfRangeException("idx") : null);
            return SvmLightLib.GetFeatureVectorWeight(m_id, idx);
        }        
        // *** IDisposable interface implementation ***
        public void Dispose()
        {
            if (m_id > 0)
            {
                SvmLightLib.DeleteFeatureVector(m_id);
                m_id = -1;
            }
        }
        // *** IEnumerableList<IdxDat2<float>> interface implementation ***
        public int Count
        {
            get { return SvmLightLib.GetFeatureVectorFeatureCount(m_id); }
        }
        public IdxDat2<float> this[int idx]
        {
            get { return new IdxDat2<float>(GetFeatureId(idx), GetFeatureWeight(idx)); } // throws ArgumentOutOfRangeException
        }
        object IEnumerableList.this[int idx]
        {
            get { return this[idx]; } // throes ArgumentOutOfRangeException
        }
        public IEnumerator<IdxDat2<float>> GetEnumerator()
        {
            return new ListEnum<IdxDat2<float>>(this);
        }
        IEnumerator IEnumerable.GetEnumerator()
        {
            return new ListEnum(this);
        }
    }
}