/*==========================================================================;
 *
 *  (c) 2004-08 JSI.  All rights reserved.
 *
 *  File:          BinarySerializer.cs
 *  Version:       1.0
 *  Desc:		   Binary serializer
 *  Author:        Miha Grcar
 *  Created on:    Oct-2004
 *  Last modified: May-2008
 *  Revision:      May-2008
 *
 ***************************************************************************/

using System;
using System.Runtime.InteropServices;
using System.Collections.Generic;
using System.Reflection;
using System.Text;
using System.IO;

namespace Latino
{
	/* .-----------------------------------------------------------------------
   	   |
	   |  Class BinarySerializer
	   |
	   '-----------------------------------------------------------------------
	*/
	public class BinarySerializer
	{
        private static Dictionary<string, string> m_full_to_short_tn
            = new Dictionary<string, string>();
        private static Dictionary<string, string> m_short_to_full_tn
            = new Dictionary<string, string>();
        private Stream m_stream;
        private string m_data_dir
            = ".";
        private static void RegisterShortTypeName(string full_name, string short_name)
        {
            m_full_to_short_tn.Add(full_name, short_name);
            m_short_to_full_tn.Add(short_name, full_name);
        }
        private static string GetFullTypeName(string short_name)
        {
            return m_short_to_full_tn.ContainsKey(short_name) ? m_short_to_full_tn[short_name] : short_name;
        }
        private static string GetShortTypeName(string full_name)
        {
            return m_full_to_short_tn.ContainsKey(full_name) ? m_full_to_short_tn[full_name] : full_name;
        }
        static BinarySerializer()
        {
            RegisterShortTypeName(typeof(bool).AssemblyQualifiedName, "b");
            RegisterShortTypeName(typeof(byte).AssemblyQualifiedName, "ui1");
            RegisterShortTypeName(typeof(sbyte).AssemblyQualifiedName, "i1");
            RegisterShortTypeName(typeof(char).AssemblyQualifiedName, "c");
            RegisterShortTypeName(typeof(double).AssemblyQualifiedName, "f8");
            RegisterShortTypeName(typeof(float).AssemblyQualifiedName, "f4");
            RegisterShortTypeName(typeof(int).AssemblyQualifiedName, "i4");
            RegisterShortTypeName(typeof(uint).AssemblyQualifiedName, "ui4");
            RegisterShortTypeName(typeof(long).AssemblyQualifiedName, "i8");
            RegisterShortTypeName(typeof(ulong).AssemblyQualifiedName, "ui8");
            RegisterShortTypeName(typeof(short).AssemblyQualifiedName, "i2");
            RegisterShortTypeName(typeof(ushort).AssemblyQualifiedName, "ui2");
            RegisterShortTypeName(typeof(string).AssemblyQualifiedName, "s");
        }
 		public BinarySerializer(Stream stream)
		{
            Utils.ThrowException(stream == null ? new ArgumentNullException("stream") : null);
			m_stream = stream;
		}
        public BinarySerializer()
        {
            m_stream = new MemoryStream();
        }
        public BinarySerializer(string file_name, FileMode file_mode)
        {
            m_stream = new FileStream(file_name, file_mode); // throws exceptions
        }
		// *** Reading ***
        private byte[] Read<T>() // Read<T> is directly or indirectly called in several methods thus exceptions thrown herein can also be thrown in all those methods
        {
            int sz = Marshal.SizeOf(typeof(T));
            byte[] buffer = new byte[sz];
            int num_bytes = m_stream.Read(buffer, 0, sz); // throws exceptions
            Utils.ThrowException(num_bytes < sz ? new EndOfStreamException() : null);
            return buffer;
        }
        public bool ReadBool()
        {
            return ReadByte() != 0;
        }
        public byte ReadByte() // ReadByte is directly or indirectly called in several methods thus exceptions thrown herein can also be thrown in all those methods
        {
            int val = m_stream.ReadByte(); // throws NotSupportedException, ObjectDisposedException
            Utils.ThrowException(val < 0 ? new EndOfStreamException() : null);
            return (byte)val;
        }
        public sbyte ReadSByte()
        {
            return (sbyte)ReadByte();
        }
        private char ReadChar8()
        {
            return (char)ReadByte();
        }
        private char ReadChar16()
        {
            return BitConverter.ToChar(Read<ushort>(), 0);
        }
        public char ReadChar()
        {
            return ReadChar16();
        }
        public double ReadDouble()
        {
            return BitConverter.ToDouble(Read<double>(), 0);
        }
        public float ReadFloat()
        {
            return BitConverter.ToSingle(Read<float>(), 0);
        }
        public int ReadInt()
        {
            return BitConverter.ToInt32(Read<int>(), 0);
        }
        public uint ReadUInt()
        {
            return BitConverter.ToUInt32(Read<uint>(), 0);
        }
        public long ReadLong()
        {
            return BitConverter.ToInt64(Read<long>(), 0);
        }
        public ulong ReadULong()
        {
            return BitConverter.ToUInt64(Read<ulong>(), 0);
        }
        public short ReadShort()
		{
            return BitConverter.ToInt16(Read<short>(), 0);
		}
        public ushort ReadUShort()
        {
            return BitConverter.ToUInt16(Read<ushort>(), 0);
        }
        private string ReadString8()
        {
            int len = ReadInt();
            if (len < 0) { return null; }
            byte[] buffer = new byte[len];
            m_stream.Read(buffer, 0, len); // throws exceptions
            return Encoding.ASCII.GetString(buffer);
        }
        private string ReadString16()
        {
            int len = ReadInt();
            if (len < 0) { return null; }
            byte[] buffer = new byte[len * 2];
            m_stream.Read(buffer, 0, len * 2); // throws exceptions
            return Encoding.Unicode.GetString(buffer);
        }
        public string ReadString()
        {
            return ReadString16();
        }
        public ValueType ReadValue(Type type)
        {
            Utils.ThrowException(type == null ? new ArgumentNullException("type") : null);
            Utils.ThrowException(!type.IsValueType ? new InvalidArgumentValueException("type") : null);
            if (type == typeof(bool))
            {
                return ReadBool();
            }
            else if (type == typeof(byte))
            {
                return ReadByte();
            }
            else if (type == typeof(sbyte))
            {
                return ReadSByte();
            }
            else if (type == typeof(char))
            {
                return ReadChar();
            }
            else if (type == typeof(double))
            {
                return ReadDouble();
            }
            else if (type == typeof(float))
            {
                return ReadFloat();
            }
            else if (type == typeof(int))
            {
                return ReadInt();
            }
            else if (type == typeof(uint))
            {
                return ReadUInt();
            }
            else if (type == typeof(long))
            {
                return ReadLong();
            }
            else if (type == typeof(ulong))
            {
                return ReadULong();
            }
            else if (type == typeof(short))
            {
                return ReadShort();
            }
            else if (type == typeof(ushort))
            {
                return ReadUShort();
            }
            else if (typeof(ISerializable).IsAssignableFrom(type))
            {
                ConstructorInfo ctor = type.GetConstructor(new Type[] { typeof(BinarySerializer) });
                Utils.ThrowException(ctor == null ? new ArgumentNotSupportedException("type") : null);
                return (ValueType)ctor.Invoke(new object[] { this }); // throws exceptions
            }
            else
            {
                throw new ArgumentNotSupportedException("type");
            }
        }
        public T ReadValue<T>()
        {
            return (T)(object)ReadValue(typeof(T)); // throws exceptions
        }
        public object ReadObject(Type type)
        {
            Utils.ThrowException(type == null ? new ArgumentNullException("type") : null);
            switch (ReadByte())
            {
                case 0:
                    return null;
                case 1:
                    break;
                case 2:
                    Type type_0 = ReadType(); // throws TargetInvocationException, ArgumentException, TypeLoadException
                    Utils.ThrowException(type_0 == null ? new InvalidOperationException() : null); // !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
                    Utils.ThrowException(!type.IsAssignableFrom(type_0) ? new InvalidArgumentValueException("type") : null);
                    type = type_0;
                    break;
                default:
                    throw new InvalidDataException();
            }
            if (type == typeof(string))
            {
                return ReadString();
            }
            else if (typeof(ISerializable).IsAssignableFrom(type))
            {
                ConstructorInfo ctor = type.GetConstructor(new Type[] { typeof(BinarySerializer) });
                Utils.ThrowException(ctor == null ? new ArgumentNotSupportedException("type") : null);
                return ctor.Invoke(new object[] { this }); // throws exceptions
            }
            else if (type.IsValueType)
            {
                return ReadValue(type); // throws exceptions
            }
            else
            {
                throw new InvalidArgumentValueException("type");
            }
        }
        public T ReadObject<T>()
        {
            return (T)ReadObject(typeof(T)); // throws exceptions
        }
        public object ReadValueOrObject(Type type)
        {
            Utils.ThrowException(type == null ? new ArgumentNullException("type") : null);
            if (type.IsValueType)
            {
                return ReadValue(type); // throws exceptions
            }
            else
            {
                return ReadObject(type); // throws exceptions
            }
        }
        public T ReadValueOrObject<T>()
        {
            return (T)ReadValueOrObject(typeof(T)); // throws exceptions
        }
        public Type ReadType()
        {
            return Type.GetType(GetFullTypeName(ReadString8())); // throws TargetInvocationException, ArgumentException, TypeLoadException
        }
		// *** Writing ***
        private void Write(byte[] data) // Write is directly or indirectly called in several methods thus exceptions thrown herein can also be thrown in all those methods
        {
            m_stream.Write(data, 0, data.Length); // throws exceptions
        }
        public void WriteBool(bool val)
        {
            WriteByte(val ? (byte)1 : (byte)0);
        }
        public void WriteByte(byte val) // WriteByte is directly or indirectly called in several methods thus exceptions thrown herein can also be thrown in all those methods
        {
            m_stream.WriteByte(val); // throws IOException, NotSupportedException, ObjectDisposedException
        }
        public void WriteSByte(sbyte val)
        {
            WriteByte((byte)val);
        }
        private void WriteChar8(char val)
        {
            WriteByte(Encoding.ASCII.GetBytes(new char[] { val })[0]);
        }
        private void WriteChar16(char val)
        {
            Write(BitConverter.GetBytes((ushort)val));
        }
        public void WriteChar(char val)
        {
            WriteChar16(val);
        }
        public void WriteDouble(double val)
        {
            Write(BitConverter.GetBytes(val));
        }
        public void WriteFloat(float val)
        {
            Write(BitConverter.GetBytes(val));
        }
		public void WriteInt(int val)
		{
            Write(BitConverter.GetBytes(val));
		}
        public void WriteUInt(uint val)
        {
            Write(BitConverter.GetBytes(val));
        }
        public void WriteLong(long val)
        {
            Write(BitConverter.GetBytes(val));
        }
        public void WriteULong(ulong val)
        {
            Write(BitConverter.GetBytes(val));
        }
        public void WriteShort(short val)
        {
            Write(BitConverter.GetBytes(val));
        }
        public void WriteUShort(ushort val)
        {
            Write(BitConverter.GetBytes(val));
        }
        private void WriteString8(string val)
        {
            if (val == null) { WriteInt(-1); return; }
            WriteInt(val.Length);
            Write(Encoding.ASCII.GetBytes(val));
        }
        private void WriteString16(string val)
        {
            if (val == null) { WriteInt(-1); return; }
            WriteInt(val.Length);
            Write(Encoding.Unicode.GetBytes(val));
        }
        public void WriteString(string val)
        {
            WriteString16(val);
        }
        public void WriteValue(ValueType val)
        {
            if (val is bool)
            {
                WriteBool((bool)val);
            }
            else if (val is byte)
            {
                WriteByte((byte)val);
            }
            else if (val is sbyte)
            {
                WriteSByte((sbyte)val);
            }
            else if (val is char)
            {
                WriteChar((char)val);
            }
            else if (val is double)
            {
                WriteDouble((double)val);
            }
            else if (val is float)
            {
                WriteFloat((float)val);
            }
            else if (val is int)
            {
                WriteInt((int)val);
            }
            else if (val is uint)
            {
                WriteUInt((uint)val);
            }
            else if (val is long)
            {
                WriteLong((long)val);
            }
            else if (val is ulong)
            {
                WriteULong((ulong)val);
            }
            else if (val is short)
            {
                WriteShort((short)val);
            }
            else if (val is ushort)
            {
                WriteUShort((ushort)val);
            }
            else if (val is ISerializable)
            {
                ((ISerializable)val).Save(this); // throws exceptions
            }
            else
            {
                throw new ArgumentTypeException("val");
            }
        }
        public void WriteObject(Type type, object obj)
        {
            Utils.ThrowException(type == null ? new ArgumentNullException("type") : null);
            Utils.ThrowException((obj != null && !type.IsAssignableFrom(obj.GetType())) ? new ArgumentTypeException("obj") : null);
            if (obj == null)
            {
                WriteByte(0);
            }
            else
            {
                Type obj_type = obj.GetType();
                if (obj_type == type)
                {
                    WriteByte(1);
                }
                else
                {
                    WriteByte(2);
                    WriteType(obj_type);
                }
                if (obj is string)
                {
                    WriteString((string)obj);
                }
                else if (obj is ISerializable)
                {
                    ((ISerializable)obj).Save(this); // throws exceptions
                }
                else if (obj is ValueType)
                {
                    WriteValue((ValueType)obj); // throws exceptions
                }
                else
                {
                    throw new ArgumentTypeException("obj");
                }
            }
        }
        public void WriteObject<T>(T obj)
        {
            WriteObject(typeof(T), obj); // throws exceptions
        }
        public void WriteValueOrObject(Type type, object obj)
        {
            Utils.ThrowException(type == null ? new ArgumentNullException("type") : null);
            Utils.ThrowException(!type.IsAssignableFrom(obj.GetType()) ? new ArgumentTypeException("obj") : null);
            if (type.IsValueType)
            {
                WriteValue((ValueType)obj); // throws exceptions
            }
            else
            {
                WriteObject(type, obj); // throws exceptions
            }
        }
        public void WriteValueOrObject<T>(T obj)
        {
            WriteValueOrObject(typeof(T), obj);
        }
        public void WriteType(Type type)
        {
            WriteString8(GetShortTypeName(type.AssemblyQualifiedName));
        }
        // *** Data directory ***
        public string DataDir
        {
            get { return m_data_dir; }
            set 
            {
                Utils.ThrowException(!Utils.VerifyPathName(value, /*must_exist=*/true) ? new InvalidArgumentValueException("DataDir setter value") : null);
                m_data_dir = value;
            }
        }
		// *** Access to the associated stream ***
        public void Close()
        {
            m_stream.Close();
        }
        public void Flush()
        {
            m_stream.Flush(); // throws IOException
        }
        public Stream Stream
        {
            get { return m_stream; }
        }
	}
}